/*
 * FullUndiGraph.h
 *
 *  Created on: 13.1.2012
 *      Author: hartman
 */

#ifndef FULLUNDIGRAPH_H_
#define FULLUNDIGRAPH_H_

#include "UndiGraph.h"
#include "SymMatrix.h"
#include <string>

class FullUndiGraph: public UndiGraph {
public:

	/*!
	 * 	Basic constructor that uses simple initiation with adjacency matrix. Input parameter also
	 * 	include size of graph
	 *
	 * 	@param mat Adjacency matrix of graph in symmetric format
	 */
	FullUndiGraph(SymMatrix<bool> *mat);

	/*!
	 * 	Constructor that uses simple initiation with adjacency matrix stored in triangular matrix.
	 *
	 * 	@param mat Adjacency matrix of graph in symmetric format
	 * 	@param n Size of corresponding graph
	 */
	FullUndiGraph(bool *mat, int n);

	/*!
	 * 	Constructor that uses simple initiation with adjacency matrix stored in triangular integer matrix.
	 * 	Internally it is transformed into boolean.
	 *
	 * 	@param mat Adjacency matrix of graph in symmetric format in integer format
	 * 	@param n Size of corresponding graph
	 */
	FullUndiGraph(int *mat, int n);

	/*!
	 * 	Initial settings of this graph after instantiation.
	 */
	void initialSettings();

	/*!
	 * 	Destructor that removes data allocated by object. Among other these are ajdacency matrix in
	 * 	::data.
	 */
	virtual ~FullUndiGraph();

	/*!
	 * 	Method to update structure of the graph by given adjacency matrix in full or triangular format.
	 *
	 * 	@param mat Adjacency matrix of graph
	 * 	@param n Size of corresponding graph
	 * 	@param triu Set to true if mat is triangular, false otherwise
	 */
	void updateStructure(bool *mat, int n, bool triu);

	/*!
	 * 	Size of the graph in sense of number of edges in its domain.
	 *
	 * @return Number of edges in this graph.
	 */
	int numberOfEdges();

	/*!
	 * Number edges adjacent to each of the vertex regardless of their possible orientation or weights.
	 *
	 * @return Degrees, i.e. number of adjacent edges to each vertex.
	 */
	int* degrees(int** degreesData = NULL, int saveData = true);

	/*!
	 * Betweenness centrality for each vertex of a graph.
	 *
	 * @param betweennessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Betweenness centrality for all vertices.
	 */
	double* betweenness(double** betweennessData = NULL, int saveData = RES_SAVED);

	/*!
	 * Closeness centrality for each vertex of a graph.
	 *
	 * @param closenessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	double* closeness(double** closenessData = NULL, int saveData = RES_SAVED);

	/*!
	 * Average Nearest Neighbor for each vertex of a graph.
	 *
	 * @param annData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	double* averageNearestNeighbor(double** annData = NULL, int saveData = RES_SAVED);

	/*!
	 * Get number of triangles around a specific node.
	 *
	 * @param nodeIndex Index of node for which the result is computed.
	 * @return The number of triangles.
	 */
	int numerOfTriangles(int nodeIndex);

	/*!
	 * Clustering coefficient for all vertices in the graph.
	 *
	 * @param clusteringCoeffData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Betweenness centrality for all vertices.
	 */
	double* clusteringCoefficient(double** clusteringCoeffData = NULL, int saveData = RES_SAVED);

	/*!
	 * Global value (average over all vertices) of clustering coefficient of the graph.
	 *
	 * @return Clustering coefficient of the graph.
	 */
	double clusteringCoefficientGlobal();

	/*!
	 * 	Transitivity of the graph.
	 *
	 * @return Transitivity of graph.
	 */
	double transitivity();

	/*!
	 * 	Method to get number of components in corresponding graph.
	 *
	 * @param componentsListData Data list into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return List of components - list of present vertices.
	 */
	vector<vector<int> >* components(vector<vector<int> >** componentsListData = NULL, int saveData = RES_SAVED);

	/*!
	 * 	Method to get components in corresponding graph in form of matrix. It cannot be pass as parameter
	 * 	because exact size of the matrix is unknown in advance.
	 *
	 * @return Matrix of components - components versus present vertices.
	 */
	int* componentsMat();

	/*!
	 * 	Method to get check whether graph is connected of not. It starts the computation as for connected components
	 * 	but it saves no data and breaks when there are more than one component.
	 *
	 * @return True if graph is connected, false otherwise.
	 */
	bool connected();

	/*!
	 * 	Method to compute number of components of the graph. It starts the computation as for connected components
	 * 	but it saves no data.
	 *
	 * @return True if graph is connected, false otherwise.
	 */
	int componentsNum();

	/**
	 * 	Method to generate random graph of Erdos Renyi type of this particular
	 *
	 * 	@param num Number of nodes of correpoding graph.
	 * 	@param p Probability with which edges of correponding graph are generated.
	 * 	@return Erdos-Renyi random graph G(num,p).
	 * */
	static FullUndiGraph *sampleErdosRenyi(int num, double p);

	/**
	 * 	Method to compute distance matrix of corresponding graph.
	 *
	 * 	@param dmatrix Reference to a pointer to distance matrix. Input/output distance matrix in first parameter
	 * 	can contain anything, i.e. best way to create is through D = new SymMatrix<int>(n), where n is corresponding
	 * 	size.
	 * 	@param saveData Save data also into instance variable for future purposes.
	 * 	@return Distance matrix same as in dmatrix parameter.
	 * */
	SymMatrix<int>* distanceMatrix(SymMatrix<int>* dmatrix = NULL, int saveData = RES_SAVED);

	/*!
	 * 	Characteristic path length computed as mean of nonzero distance matrix elements. Contains computation
	 * 	of distance matrix.
	 *
	 * @return Characteristic path length.
	 */
	double characterticPathLength();

	/*!
	 * 	Efficiency of graph computed as mean of nonzero distance matrix elements. Contains computation
	 * 	of distance matrix.
	 *
	 * @return Efficiency.
	 */
	double efficiency();

	/*!
	 * Method to generate complete graph of size m.
	 *
	 *	@param m Size of newly generated complete graph.
	 * 	@return Complete graph of size m.
	 */
	static FullUndiGraph* K_m(int m);

	/*!
	 * 	Assortative coefficient of the graph.
	 *
	 * @return Assortative coefficient.
	 */
	double assortativeCoefficient();


protected:
	/*! adjacency matrix */
	SymMatrix<bool> data;

	/*! adjacency matrix */
	SymMatrix<int> *D;

	/*! indication that distance matrix has been accessed */
	char_state_t dGet;

	/*! Number of edges */
	int m;


	/*!
	 * 	Method that is called if the structure of graph changes and so corresponding characteristics are no longer valid.
	 */
	void changeState();

	FullUndiGraph();

	void visit(vector<vector<int> > *cmps, int i);

	void visit(int i);

	string to_s();
};

#endif /* FULLUNDIGRAPH_H_ */
