/*
 * Graph.h
 *
 *  Created on: 13.1.2012
 *      Author: hartman
 */

#ifndef GRAPH_H_
#define GRAPH_H_
#include "stddef.h"
#include <string>
#include <cstring>
#include "SymMatrix.h"
#include "TypesDefs.h"

using namespace std;

/** Macro to get input state from data, vector existence */
#define get_input_state(data,vector) (data == NULL) ? ((vector == NULL) ? INPUT_NONE:INPUT_VECTOR):((vector == NULL) ? INPUT_DATA:INPUT_BOTH);


class Graph {
public:
	/*!
	 * 	Constructor of graph. While Graph is a general class it is just for childs. It only creates and sets ::testingDoubleVector and ::testingIntVector
	 */
	Graph();

	/*!
	 * 	Destructor of a Graph.
	 */
	virtual ~Graph();

	/*!
	 * 	Size of the graph in sense of number of vertices in its domain.
	 *
	 * @return Number of underlying vertices.
	 */
	int size();

	/*!
	 * 	Size of the graph in sense of number of edges in its domain.
	 *
	 * @return Number of edges in this graph.
	 */
	virtual int numberOfEdges() = 0;

	/*!
	 * 	Maximum number of edges this particular graph can have (depends on directionality, multiplicity etc.)
	 *
	 * @return Maximum number of edges in this graph.
	 */
	virtual int maxNumberOfEdges() = 0;

	/*!
	 * Density as number of edges relatively to maximum number of edges.
	 *
	 * @return Density of the graph.
	 */
	double density();

	/*!
	 * Get number of triangles around a specific node.
	 *
	 * @param nodeIndex Index of node for which the result is computed.
	 * @return The number of triangles.
	 */
	virtual int numerOfTriangles(int nodeIndex) = 0;

	/*!
	 * Clustering coefficient for all vertices in the graph.
	 *
	 * @param clusteringCoeffData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Clustering coefficient for particular vertices of graph.
	 */
	virtual double* clusteringCoefficient(double** clusteringCoeffData = NULL, int saveData = RES_SAVED) = 0;

	/*!
	 * Global value (average over all vertices) of clustering coefficient of the graph.
	 *
	 * @return Clustering coefficient of the graph.
	 */
	virtual double clusteringCoefficientGlobal() = 0;

	/*!
	 * 	Transitivity of the graph.
	 *
	 * @return Transitivity of graph.
	 */
	virtual double transitivity() = 0;

	/*!
	 * 	Method to get components in corresponding graph in form of list.
	 *
	 * @param componentsListData List into which results should be saved, if NULL only returning
	 * reference is used. First dimension is components and second is for participating vertices.
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return List of components - list of present vertices.
	 */
	virtual vector<vector<int> >* components(vector<vector<int> >** componentsListData = NULL, int saveData = RES_SAVED) = 0;

	/*!
	 * 	Method to get components in corresponding graph in form of matrix. It cannot be pass as parameter
	 * 	because exact size of the matrix is unknown in advance.
	 *
	 * @return Matrix of components - components versus present vertices.
	 */
	virtual int* componentsMat() = 0;

	/*!
	 * Betweenness centrality for each vertex of a graph.
	 *
	 * @param betweennessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Betweenness centrality for all vertices.
	 */
	virtual double* betweenness(double** betweennessData = NULL, int saveData = RES_SAVED) = 0;

	/*!
	 * Closeness centrality for each vertex of a graph.
	 *
	 * @param closenessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	virtual double* closeness(double** closenessData = NULL, int saveData = RES_SAVED) = 0;

	/*!
	 * Average Nearest Neighbor for each vertex of a graph.
	 *
	 * @param annData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	virtual double* averageNearestNeighbor(double** annData = NULL, int saveData = RES_SAVED) = 0;

	/**
	 * 	Method to compute distance matrix of corresponding graph in sense of number of edges (no possible weights considered).
	 *
	 * 	@param dmatrix Reference to a pointer to distance matrix. Input/output distance matrix in first parameter
	 * 	can contain anything, i.e. best way to create is through D = new SymMatrix<int>(n), where n is corresponding
	 * 	size.
	 * 	@param saveData Save data also into instance variable for future purposes.
	 * 	@return Distance matrix same as in dmatrix parameter.
	 * */
	virtual SymMatrix<int>* distanceMatrix(SymMatrix<int>* dmatrix = NULL, int saveData = RES_SAVED) = 0;

	/*!
	 * 	Characteristic path length computed as mean of nonzero distance matrix elements. Contains computation
	 * 	of distance matrix.
	 *
	 * @return Characteristic path length.
	 */
	virtual double characterticPathLength() = 0;

	/*!
	 * 	Efficiency of graph computed as mean of nonzero distance matrix elements. Contains computation
	 * 	of distance matrix.
	 *
	 * @return Efficiency.
	 */
	virtual double efficiency() = 0;

	/*!
	 * 	Method to compute number of components of the graph. It starts the computation as for connected components
	 * 	but it saves no data.
	 *
	 * @return True if graph is connected, false otherwise.
	 */
	virtual int componentsNum() = 0;

	/*!
	 * 	Assortative coefficient of the graph.
	 *
	 * @return Assortative coefficient.
	 */
	virtual double assortativeCoefficient() = 0;

	/*!
	 * 	Testing method that shows how simple double vector is created and returned. All characteristics of
	 * 	vector type should be processed this way.
	 *
	 * 	@param data Reference to array with characteristics
	 * 	@param saveData saveData state variable possibly having RES_SAVED,RES_COPY_SAVE, RES_NOT_SAVED
	 * 	@return Results of the method (possibly same as in data - if not NULL at the beginning)
	 */
	double* plainDoubleReturingMethod(double** data = NULL, int saveData = RES_SAVED);

	/*!
	 * 	Testing method that shows how simple int vector is created and returned. All characteristics of
	 * 	vector type should be processed this way.
	 *
	 * 	@param data Reference to array with characteristics
	 * 	@param saveData saveData state variable possibly having RES_SAVED,RES_COPY_SAVE, RES_NOT_SAVED
	 * 	@return Results of the method (possibly same as in data - if not NULL at the beginning)
	 */
	int* plainIntReturingMethod(int** data = NULL, int saveData = RES_SAVED);

protected:
	/** Size of graph */
	int n;

	/*! Testing double vector */
	double* testingDoubleVector;
	/*! Testing double vector get attribute */
	char_state_t testingDoubleVectorGet;

	/*! Testing int vector */
	int* testingIntVector;
	/*! Testing int vector get attribute */
	char_state_t testingIntVectorGet;

	/*! Vector containing shortest path betweeness centrality of corresponding graph */
	double* betweennessVector;
	/*! Indicator of reponsibility for shortest path betweeness centrality. If true vector has been
	 * accessed from outside and responsibility for its deletion goes there. */
	char_state_t betweennessVectorGet;

	/*! Vector containing shortest path closeness centrality of corresponding graph */
	double* closenessVector;
	/*! Indicator of reponsibility for shortest path closeness centrality. If true vector has been
	 * accessed from outside and responsibility for its deletion goes there. */
	char_state_t closenessVectorGet;

	/*! Vector containing shortest path closeness centrality of corresponding graph */
	double* annVector;
	/*! Indicator of reponsibility for shortest path closeness centrality. If true vector has been
	 * accessed from outside and responsibility for its deletion goes there. */
	char_state_t annVectorGet;

	/*! Vector containing clustering coefficient for all vertices of the corresponding graph */
	double* clusteringCoeffVector;
	/*! Indicator of reponsibility for clustering coefficient. If true vector has been
	 * accessed from outside and responsibility for its deletion goes there. */
	char_state_t clusteringCoeffVectorGet;

	/*! List of connected components, i.e. list of lists of vertices present in particular compoenents. */
	vector<vector<int> >* componentsList;
	/*! auxiliary field for connected components determination */
	bool* visited;
	/*! auxiliary value for connected components determination */
	int componentIndex;
	/*! Indicator of reponsibility for components. If true vector has been
	 * accessed from outside and responsibility for its deletion goes there. */
	char_state_t componentsGet;

	/*!
	 * 	Method that is called if the structure of graph changes and so corresponding characteristics are no longer valid.
	 */
	void changeState();


	/**
	 * Method to correctly store data into output array and (possibly) internal vector depending
	 * on input state of data storages (data and vector) and parameter 'save'.
	 *
	 * 	@param input_state Parameter for coding state of data storage inputs into algorithms. Possible values
	 * 	are defined by INPUT_NONE, INPUT_VECTOR, INPUT_DATA and INPUT_BOTH.
	 * 	@param save Parameter to indicate ways to save data. Possible values are: RES_NOT_SAVED, RES_SAVED
	 * 	and RES_COPY_SAVED.
	 * 	@param data Reference to pointer to data storage that will be used to store results of computation.
	 * 	@param vector Reference to pointer to internal data containing results of (possibly previous) computation.
	 * 	@param vectorGet reference to indicator variable used to determine whether corresponding vector has been
	 * 	obtained from outside as reference and thus all responsibility is on their side.
	 * 	@param n Size of considered data vector
	 * 	@return Data results same as those saved in data parameter.
	 */
	double* close_double_method(int input_state, int save, double** data, double* &vector, char_state_t &vectorGet, int n);

	/**
	 * Method to correctly store data into output array and (possibly) internal vector depending
	 * on input state of data storages (data and vector) and parameter 'save'.
	 *
	 * 	@param input_state Parameter for coding state of data storage inputs into algorithms. Possible values
	 * 	are defined by INPUT_NONE, INPUT_VECTOR, INPUT_DATA and INPUT_BOTH.
	 * 	@param save Parameter to indicate ways to save data. Possible values are: RES_NOT_SAVED, RES_SAVED
	 * 	and RES_COPY_SAVED.
	 * 	@param data Reference to pointer to data storage that will be used to store results of computation.
	 * 	@param vector Reference to pointer to internal data containing results of (possibly previous) computation.
	 * 	@param vectorGet reference to indicator variable used to determine whether corresponding vector has been
	 * 	obtained from outside as reference and thus all responsibility is on their side.
	 * 	@param n Size of considered data vector
	 * 	@return Data results same as those saved in data parameter.
	 */
	int* close_int_method(int input_state, int save, int** data, int* &vector, char_state_t &vectorGet, int n);

	/**
	 * Method to correctly store data into output array and (possibly) internal vector depending
	 * on input state of data storages (data and vector) and parameter 'save'.
	 *
	 * 	@param input_state Parameter for coding state of data storage inputs into algorithms. Possible values
	 * 	are defined by INPUT_NONE, INPUT_VECTOR, INPUT_DATA and INPUT_BOTH.
	 * 	@param save Parameter to indicate ways to save data. Possible values are: RES_NOT_SAVED, RES_SAVED
	 * 	and RES_COPY_SAVED.
	 * 	@param data Reference to pointer to data storage that will be used to store results of computation.
	 * 	@param vector Reference to pointer to internal data containing results of (possibly previous) computation.
	 * 	@param vectorGet reference to indicator variable used to determine whether corresponding vector has been
	 * 	obtained from outside as reference and thus all responsibility is on their side.
	 * 	@param n Size of considered data vector
	 * 	@return Data results same as those saved in data parameter.
	 */
	SymMatrix<int>* close_symmat_int_method(int input_state, int save, SymMatrix<int>** data, SymMatrix<int>* &vector, char_state_t &vectorGet, int n);


	/**
	 * Method to correctly store data into output array and (possibly) internal vector depending
	 * on input state of data storages (data and vector) and parameter 'save'.
	 *
	 * 	@param input_state Parameter for coding state of data storage inputs into algorithms. Possible values
	 * 	are defined by INPUT_NONE, INPUT_VECTOR, INPUT_DATA and INPUT_BOTH.
	 * 	@param save Parameter to indicate ways to save data. Possible values are: RES_NOT_SAVED, RES_SAVED
	 * 	and RES_COPY_SAVED.
	 * 	@param data Reference to pointer to data storage that will be used to store results of computation.
	 * 	@param vector Reference to pointer to internal data containing results of (possibly previous) computation.
	 * 	@param vectorGet reference to indicator variable used to determine whether corresponding vector has been
	 * 	obtained from outside as reference and thus all responsibility is on their side.
	 * 	@param n Size of considered data vector
	 * 	@return Data results same as those saved in data parameter.
	 */
	vector<vector<int> >* close_vecvec_int_method(int input_state, int save, vector<vector<int> >** data, vector<vector<int> >* &vector, char_state_t &vectorGet, int n);

};

#endif /* GRAPH_H_ */
