/*
 * SparseUndiGraph.h
 *
 *  Created on: 13.1.2012
 *      Author: hartman
 */

#ifndef SPARSEUNDIGRAPH_H_
#define SPARSEUNDIGRAPH_H_
#include "UndiGraph.h"

class SparseUndiGraph: public UndiGraph {
public:

	SparseUndiGraph(int* mat, int lsize);

	SparseUndiGraph(bool* mat, int lsize);

	virtual ~SparseUndiGraph();

	/*!
	 * 	Initial settings of this graph after instantiation.
	 */
	void initialSettings();

	/*!
	 * 	Size of the graph in sense of number of edges in its domain.
	 *
	 * @return Number of edges in this graph.
	 */
	int numberOfEdges();

	/*!
	 * Number edges adjacent to each of the vertex regardless of their possible orientation or weights.
	 *
	 * @return Degrees, i.e. number of adjacent edges to each vertex.
	 */
	int* degrees(int** degreesData = NULL, int saveData = true);

	/*!
	 * Betweenness centrality for each vertex of a graph.
	 *
	 * @param betweennessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Betweenness centrality for all vertices.
	 */
	double* betweenness(double** betweennessData = NULL, int saveData = RES_SAVED);

	/*!
	 * Closeness centrality for each vertex of a graph.
	 *
	 * @param closenessData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	double* closeness(double** closenessData = NULL, int saveData = RES_SAVED);

	/*!
	 * Average Nearest Neighbor for each vertex of a graph.
	 *
	 * @param annData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Closeness centrality for all vertices.
	 */
	double* averageNearestNeighbor(double** annData = NULL, int saveData = RES_SAVED);


	/*!
	 * Clustering coefficient for all vertices in the graph.
	 *
	 * @param clusteringCoeffData Data vector into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return Betweenness centrality for all vertices.
	 */
	double* clusteringCoefficient(double** clusteringCoeffData = NULL, int saveData = RES_SAVED);

	/*!
	 * Global value (average over all vertices) of clustering coefficient of the graph.
	 *
	 * @return Clustering coefficient of the graph.
	 */
	double clusteringCoefficientGlobal();

	/*!
	 * 	Transitivity of the graph.
	 *
	 * @return Transitivity of graph.
	 */
	double transitivity();

	/*!
	 * 	Method to get number of components in corresponding graph.
	 *
	 * @param componentsListData Data list into which results should be saved, if NULL only returning
	 * reference is used
	 * @param saveData Save data also into instance variable for future purposes.
	 * @return List of components - list of present vertices.
	 */
	vector<vector<int> >* components(vector<vector<int> >** componentsListData = NULL, int saveData = RES_SAVED);

	/*!
	 * 	Method to get components in corresponding graph in form of matrix. It cannot be pass as parameter
	 * 	because exact size of the matrix is unknown in advance.
	 *
	 * @return Matrix of components - components versus present vertices.
	 */
	int* componentsMat();

	/*!
	 * Get number of triangles around a specific node.
	 *
	 * @param nodeIndex Index of node for which the result is computed.
	 * @return The number of triangles.
	 */
	int numerOfTriangles(int nodeIndex);

	/** Method to create string representing its internal adjacency matrix.
	 *
	 * 	@return Text representation of adjacency matrix.
	 **/
	string to_s();

	/**
	 * 	Method to generate random graph of Erdos Renyi type of this particular
	 *
	 * 	@param num Number of nodes of correpoding graph.
	 * 	@param p Probability with which edges of correponding graph are generated.
	 * 	@return Erdos-Renyi random graph G(num,p).
	 * */
	static SparseUndiGraph *sampleErdosRenyi(int num, double p);

	/*!
	 * Method to generate complete graph of size m.
	 *
	 *	@param m Size of newly generated complete graph.
	 * 	@return Complete graph of size m.
	 */
	SparseUndiGraph* K_m(int m);

	/**
	 * 	Method to compute distance matrix of corresponding graph.
	 *
	 * 	@param dmatrix Reference to a pointer to distance matrix. Input/output distance matrix in first parameter
	 * 	can contain anything, i.e. best way to create is through D = new SymMatrix<int>(n), where n is corresponding
	 * 	size.
	 * 	@param saveData Save data also into instance variable for future purposes.
	 * 	@return Distance matrix same as in dmatrix parameter.
	 * */
	SymMatrix<int>* distanceMatrix(SymMatrix<int>* dmatrix = NULL, int saveData = RES_SAVED);

	/*!
	 * 	Characteristic path length computed as mean of nonzero distance matrix elements.
	 *
	 * @return Characteristic path length.
	 */
	double characterticPathLength();

	/*!
	 * 	Efficiency of graph computed as mean of nonzero distance matrix elements. Contains computation
	 * 	of distance matrix.
	 *
	 * @return Efficiency.
	 */
	double efficiency();

	/*!
	 * 	Method to compute number of components of the graph. It starts the computation as for connected components
	 * 	but it saves no data.
	 *
	 * @return True if graph is connected, false otherwise.
	 */
	int componentsNum();

	/*!
	 * 	Assortative coefficient of the graph.
	 *
	 * @return Assortative coefficient.
	 */
	double assortativeCoefficient();

protected:
	int m;
	int** neighs;

	/*! adjacency matrix */
	SymMatrix<int> *D;
	/*! indication that distance matrix has been accessed */
	char_state_t dGet;

	SparseUndiGraph();

};

#endif /* SPARSEUNDIGRAPH_H_ */
