/*
 * SparseUndiGraph.cpp
 *
 *  Created on: 13.1.2012
 *      Author: hartman
 */

#include "SparseUndiGraph.h"
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <ctime>
#include <queue>
#include <vector>
#include <list>
#include <stack>
#include <algorithm>

//#define DEBUG false
//#define INDICATION true

SparseUndiGraph::SparseUndiGraph(int* mat, int lsize) {
	n = lsize;
	m = 0;

	neighs = new int*[n];
	degreesVector = new int[n];
	int* aux = new int[n];

	for (int i = 0; i < n; i++)
	{
		// check for neighbors
		int idx = 0;
		int pos = i * n;
		for (int j = 0; j < n; j++)
			if (mat[pos + j] == 1)
				aux[idx++] = j;

		// sum up edges
		m = m + idx;

		// store neighbors
		degreesVector[i] = idx;
		if (idx > 0)
		{
			neighs[i] = new int[idx];
			memcpy(neighs[i],aux,idx*sizeof(int));
		}
		else
		{
			neighs[i] = NULL;
		}
	}

	// correct edges for twice summation
	m = m / 2;

	delete aux;

}

SparseUndiGraph::SparseUndiGraph(bool* mat, int lsize) {
	n = lsize;
	m = 0;

	neighs = new int*[n];
	degreesVector = new int[n];
	int* aux = new int[n];

	for (int i = 0; i < n; i++)
	{
		// check for neighbors
		int idx = 0;
		int pos = i * n;
		for (int j = 0; j < n; j++)
			if (mat[pos + j])
				aux[idx++] = j;

		// sum up edges
		m = m + idx;

		// store neighbors
		degreesVector[i] = idx;
		if (idx > 0)
		{
			neighs[i] = new int[idx];
			memcpy(neighs[i],aux,idx*sizeof(int));
		}
		else
		{
			neighs[i] = NULL;
		}
	}

	// correct edges for twice summation
	m = m / 2;

	delete aux;

}


SparseUndiGraph::SparseUndiGraph() {
}

void SparseUndiGraph::initialSettings()
{
	D = NULL;
	dGet = false;
}


SparseUndiGraph::~SparseUndiGraph() {
	if (neighs != NULL)
	{
		for (int i = 0; i < n; i++)
		{
			delete [] neighs[i];
		}
		delete [] neighs;
	}
	if (D != NULL && ((dGet & CHAR_STATE_GET) != CHAR_STATE_GET))
		delete D;
}

int SparseUndiGraph::numberOfEdges()
{
	return m;
}



int *SparseUndiGraph::degrees(int** degreesData, int saveData)
{
	// ----------------- Init data storage --------------------------
	// determine input state
	int input_state = get_input_state(degreesData, degreesVector);
	// create reference to array of size n
	if (input_state == INPUT_NONE)
	{
		degreesData = new int*[0];
		*degreesData = new int[n];
	}

	// ------------------ computation part --------------------------
	// degrees are computed in the beginning (degreesVector has to exists), so there is no need for computation

	// ----------------- Storing data  --------------------------
	return close_int_method(input_state, saveData, degreesData, degreesVector, degreesVectorGet, size());
}

string SparseUndiGraph::to_s()
{
	std::stringstream matstr;
	for (int i = 0; i < n; i++)
	{
		int idx = 0;
		if (neighs[i] == NULL)
		{
			for (int j = 0; j < n; j++)
				matstr << "0\t";
		}
		else
		{
			for (int j = 0; j < n; j++)
			{
				if (neighs[i][idx] == j)
				{
					matstr << "1";
					idx++;
				}
				else
				{
					matstr << "0";
				}
				matstr << "\t";
			}
		}
		matstr << "\n";
	}
	return matstr.str();

}

SparseUndiGraph *SparseUndiGraph::sampleErdosRenyi(int num, double p)
{
	srand ( time(NULL) );

	// create graph
	SparseUndiGraph* sud = new SparseUndiGraph();
	sud->n = num; // set number of vertices
	int mun = 0;  // init number of edges to 0

	// create internal neighbors array and degree vector and auxiliary array
	int** parNeighs = new int*[num];
	int* parDegreeVector = new int[num];
	int* aux = new int[num];

	// random seed (joined with time for more random character)
	srand((unsigned)time(0));

	// random graph generation
	for (int i = 0; i < num; i++)
	{
		// check for neighbors
		int idx = 0;
		for (int j = 0; j < num; j++)
			if (rand()/(RAND_MAX + 1.0) < p)
				aux[idx++] = j;

		// sum up edges
		mun = mun + idx;

		// store neighbors
		parDegreeVector[i] = idx;
		if (idx > 0)
		{
			parNeighs[i] = new int[idx];
			memcpy(parNeighs[i],aux,idx*sizeof(int));
		}
		else
		{
			parNeighs[i] = NULL;
		}
	}

	// set results to new graph
	sud->degreesVector = parDegreeVector;
	sud->neighs = parNeighs;
	sud->m = mun;

	// clean sources
	delete aux;

	return sud;
}

/*!
 * Method to generate complete graph of size m.
 *
 *	@param m Size of newly generated complete graph.
 * 	@return Complete graph of size m.
 */
SparseUndiGraph* SparseUndiGraph::K_m(int m)
{
	SparseUndiGraph* sud = new SparseUndiGraph();

	int** parNeighs = new int*[m];
	int* aux = new int[m];
	for (int i = 0; i < m; i++)
		aux[i] = i;

	for (int i = 0; i < m; i++)
	{
		parNeighs[i] = new int[m];
		memcpy(parNeighs[i],aux,m*sizeof(int));
	}

	return sud;
}

double *SparseUndiGraph::betweenness(double **betweennessData, int saveData)
{
	// ----------------- Init data storage --------------------------
	// determine input state
	int input_state = get_input_state(betweennessData, betweennessVector);
	// create reference to array of size n
	if (input_state == INPUT_NONE)
	{
		betweennessData = new double*[0];
		*betweennessData = new double[n];
	}

	// ------------------ computation part --------------------------
	// if no internal vector exists compute
	if (input_state == INPUT_NONE || input_state == INPUT_DATA)
	{

			// put results into this array
			double* results = *betweennessData;
			memset(results,0,sizeof(double)*n);

			// process variables
			bc_sigma_t* sigma = new bc_sigma_t[n];
			bc_d_t* d = new bc_d_t[n];
			double* delta = new double[n];
			list<int>* P = new list<int>[n];

			// auxiliary fields with -1 and 0
			bc_d_t* d2 = new bc_d_t[n];
			std::fill(d2,d2+n,-1);
			double* delta2 = new double[n];
			memset(delta2,0,sizeof(*delta)*n);

			// stack
			int* S = new int[n];
			int sidx = 0;

			// queue
			int* Q = new int[n];
			int qstart = 0;
			int qend = 0;

			for (int s = 0; s < n; s++)
			{

				#if (INDICATION)
					cout << "\r Betweeness:" << s << "/" << n;
					cout.flush();
				#endif

				//stack<int> S;

				//memset(sigma,0,sizeof(bc_sigma_t)*n); // - not needed
				//std::fill(d,d+n,-1);
				memcpy(d,d2,n*sizeof(bc_d_t));

//				for (int j = 0; j < n; j++)
//				{
//					sigma[j] = 0;
//					d[j] = -1;
//				}
				sigma[s] = 1;
				d[s] = 0;


				//queue<int> Q;
				//Q.push(s);  // maybe push_back
				qstart = 0;
				qend = 0;
				Q[qend++] = s;

				//while (!Q.empty())
				for (int q = qstart; q < qend; q++)
				{
					//int v = Q.front();
					//Q.pop();
					int v = Q[qstart++];
					S[sidx++] = v;
					//S.push(v);

					// go through all neighbors
					int wdeg = degreesVector[v];
					int* vneighs = neighs[v];
					bc_d_t d_v = d[v];
					bc_sigma_t sigma_v = sigma[v];
					for (int wid = 0; wid < wdeg; wid++)
					{
						int w = vneighs[wid];

						if (w != v) // only neighs
						{
							if (d[w] < 0)
							{
								//Q.push(w);
								Q[qend++] = w;
								d[w] = d_v + 1;
								sigma[w] = sigma_v;
								//P[w][pix[w]++] = v;
								P[w].push_back(v);
							}
							else if (d[w] == d_v + 1)
							{
								sigma[w] += sigma_v;
								P[w].push_back(v);
								//P[w][pix[w]++] = v;
								#if (DEBUG)
									cout << " sigma[" << w << "] = " << sigma[w] << endl;
								#endif

							}
						}
					}
				}

				memcpy(delta,delta2,sizeof(double)*n);

//				for (int j = 0; j < n; j++)
//					delta[j] = 0.0;


				//while (!S.empty())
				for (int i = sidx - 1; i >= 0; i--)
				{
					//int w = S.top();
					//S.pop();
					int w = S[i];
					list<int>::iterator it;

					#if (DEBUG)
						//cout << "Prvek:" << w << endl;
					#endif

					double wcoeff = (1.0+delta[w]) / ((double)sigma[w]);


					//for (int k = 0; k<pix[w]; k++)
					for (it = P[w].begin() ; it != P[w].end(); it++)
					{
						// original
						int v = *it;  // int v = (int) *it
						delta[v] = delta[v] + sigma[v]*wcoeff;


						// int v = P[w].list[k];
						// delta[v] = delta[v] + sigma[v]*wcoeff;

						// My P
						//int v = P[w][k];  // int v = (int) *it
						//delta[v] = delta[v] + sigma[v]*wcoeff;

						#if (DEBUG)
							cout << "\t "<< delta[v] << "=" << sigma[v] << "/" << sigma[w] << "*(1 + " << delta[w] << ")" << endl;
						#endif
					}

					if (w != s)
					{
						results[w] = results[w] + delta[w];
						#if (DEBUG)
							cout << "\t Cb = "<< results[w] << endl;
						#endif
					}

					//P[w].count = 0;
				}

				// delete aux data
				for (int i = 0;  i < sidx; i++)
				{
					P[i].clear();
					// pix[w] = 0;
					// P[i].count = 0;
				}

				sidx = 0;
			}

			delete[] P;
			delete[] S;
			delete[] sigma;
			delete[] d;
			delete[] delta;
	}

	// ----------------- Storing data  --------------------------
	return close_double_method(input_state, saveData, betweennessData, betweennessVector, betweennessVectorGet, n);

// ------------------------------- LAST O.K. Version ------------------------------------------------------

//	// ----------------- Init data storage --------------------------
//	// determine input state
//	int input_state = get_input_state(betweennessData, betweennessVector);
//	// create reference to array of size n
//	if (input_state == INPUT_NONE)
//	{
//		betweennessData = new double*[0];
//		*betweennessData = new double[n];
//	}
//
//	// ------------------ computation part --------------------------
//	// if no internal vector exists compute
//	if (input_state == INPUT_NONE || input_state == INPUT_DATA)
//	{
//
//			// put results into this array
//			double* results = *betweennessData;
////			bc_aux_t* results = new bc_aux_t[n];
//			memset(results,0,sizeof(double)*n);
////			for (int i = 0; i < n; i++)
////				results[i] = 0.0;
//
//			bc_sigma_t* sigma = new bc_sigma_t[n];
//			bc_d_t* d = new bc_d_t[n];
//			double* delta = new double[n];
//			list<int>* P = new list<int>[n];
//
//			for (int s = 0; s < n; s++)
//			{
//
//				#if (INDICATION)
//					cout << "\r Betweeness:" << s << "/" << n;
//					cout.flush();
//				#endif
//
//				stack<int> S;
//
//				memset(sigma,0,sizeof(bc_sigma_t)*n);
//				std::fill(d,d+n,-1);
//
////				for (int j = 0; j < n; j++)
////				{
////					sigma[j] = 0;
////					d[j] = -1;
////				}
//				sigma[s] = 1;
//				d[s] = 0;
//
//
//				queue<int> Q;
//				Q.push(s);  // maybe push_back
//
//				while (!Q.empty())
//				{
//					int v = Q.front();
//					Q.pop();
//					S.push(v);
//
//					// go through all neighbors
//					int wdeg = degreesVector[v];
//					int* vneighs = neighs[v];
//					bc_d_t d_v = d[v];
//					bc_sigma_t sigma_v = sigma[v];
//					for (int wid = 0; wid < wdeg; wid++)
//					{
//						int w = vneighs[wid];
//
//						if (w != v) // only neighs
//						{
//							if (d[w] < 0)
//							{
//								Q.push(w);
//								d[w] = d_v + 1;
//							}
//
//							if (d[w] == d_v + 1)
//							{
//								sigma[w] += sigma_v;
//								P[w].push_back(v);
//								#if (DEBUG)
//									cout << " sigma[" << w << "] = " << sigma[w] << endl;
//								#endif
//
//							}
//						}
//					}
//				}
//
//				memset(delta,0,sizeof(*delta)*n);
//
////				for (int j = 0; j < n; j++)
////					delta[j] = 0.0;
//
//				#if (DEBUG)
////					cout << "Sigmas:" << endl;
////					for (int j = 0; j < n; j++)
////						cout << sigma[j] << " ";
////					cout << endl;
//				#endif
//
//
//				while (!S.empty())
//				{
//					int w = S.top();
//					S.pop();
//					list<int>::iterator it;
//
//					#if (DEBUG)
//						//cout << "Prvek:" << w << endl;
//					#endif
//
//					double wcoeff = (1.0+delta[w]) / ((double)sigma[w]);
//
//					for (it = P[w].begin() ; it != P[w].end(); it++)
//					{
//						int v = *it;  // int v = (int) *it
//						delta[v] = delta[v] + sigma[v]*wcoeff;
//
//						#if (DEBUG)
//							cout << "\t "<< delta[v] << "=" << sigma[v] << "/" << sigma[w] << "*(1 + " << delta[w] << ")" << endl;
//						#endif
//					}
//
//					if (w != s)
//					{
//						results[w] = results[w] + delta[w];
//						#if (DEBUG)
//							cout << "\t Cb = "<< results[w] << endl;
//						#endif
//					}
//				}
//
//				// delete aux data
//				for (int i = 0;  i < n; i++)
//					P[i].clear();
//			}
//
//			delete[] P;
//			delete sigma;
//			delete d;
//			delete delta;
//	}
//
//	// ----------------- Storing data  --------------------------
//	return close_double_method(input_state, saveData, betweennessData, betweennessVector, betweennessVectorGet, n);

// --------------------------------- LAST O.K. Version end ---------------------------------------


//	for (int i = 0; i < size; i++)
//		betweennessVector[i] = 0.0;
//
//	unsigned long* sigma = new unsigned long[size];
//	long* d = new long[size];
//	double* delta = new double[size];
//	list<int>* P = new list<int>[size];
//
//	for (int s = 0; s < size; s++)
//	{
//
//		#if (INDICATION)
//			cout << "\r Betweeness:" << s << "/" << size;
//			cout.flush();
//		#endif
//
//		stack<int> S;
//
//		for (int j = 0; j < size; j++)
//		{
//			sigma[j] = 0;
//			d[j] = -1;
//		}
//		sigma[s] = 1;
//		d[s] = 0;
//
//
//		queue<int> Q;
//		Q.push(s);  // maybe push_back
//
//		while (!Q.empty())
//		{
//			int v = Q.front();
//			Q.pop();
//			S.push(v);
//
//			// go through all neighbors
//			int wdeg = degrees[v];
//			long d_v = d[v];
//			unsigned long sigma_v = sigma[v];
//			for (int wid = 0; wid < wdeg; wid++)
//			{
//				int w = neighs[v][wid];
//				if (d[w] < 0)
//				{
//					Q.push(w);
//					d[w] = d_v + 1;
//				}
//
//				if (d[w] == d_v + 1)
//				{
//					sigma[w] = sigma[w] + sigma_v;
//					P[w].push_back(v);
//				}
//			}
//		}
//
//		for (int j = 0; j < size; j++)
//			delta[j] = 0.0;
//
//		#if (DEBUG)
//			cout << "Sigmas:" << endl;
//			for (int j = 0; j < size; j++)
//				cout << sigma[j] << " ";
//			cout << endl;
//		#endif
//
//
//		while (!S.empty())
//		{
//			int w = S.top();
//			S.pop();
//			list<int>::iterator it;
//
//			#if (DEBUG)
//				cout << "Prvek:" << w << endl;
//			#endif
//
//			double wcoeff = (1.0+delta[w]) / ((double)sigma[w]);
//
//			for (it = P[w].begin() ; it != P[w].end(); it++)
//			{
//				int v = *it;  // int v = (int) *it
//				delta[v] = delta[v] + sigma[v]*wcoeff;
//
//				#if (DEBUG)
//					cout << "\t "<< delta[v] << "=" << sigma[v] << "/" << sigma[w] << "*(1 + " << delta[w] << ")" << endl;
//					cout << "\t Cb = "<< betweennessVector[w] << endl;
//				#endif
//			}
//
//			if (w != s)
//			{
//				betweennessVector[w] = betweennessVector[w] + delta[w];
//			}
//		}
//
//		// delete aux data
//		for (int i = 0;  i < size; i++)
//			P[i].clear();
//	}
//
//	delete[] P;
//	delete sigma;
//	delete d;
//	delete delta;
//
//	double sum = 0.0;
//	for (int i = 0; i < size; i++)
//	{
//		sum += betweennessVector[i];
//	}
//
//	double result = (1/((double) size))*sum;
//
//	return result;
}

int SparseUndiGraph::numerOfTriangles(int nodeIndex)
{
	// TODO
	// auxiliary array with one on neighbor positions
	int k = degreesVector[nodeIndex];
	int* nodeNeighs = neighs[nodeIndex];

	// iterate through whole vector and get
	int sum = 0;
	for (int i = 0; i < k; i++)
	{
		int elem = nodeNeighs[i];
		int* elemNeighs = neighs[elem];
		int ek = degreesVector[elem];

		// iterate though elemNeighs and decide for each whether it is in nodeNeighs
		int idx = i+1;
		for (int j = 0; j < ek; j++)
		{
			int elemN = elemNeighs[j];
			for (; idx < k; idx++)
			{
				if (nodeNeighs[idx] == elemN)
				{
					sum += 2;
					idx++;
					break;
				}
				else if (nodeNeighs[idx] > elemN)
				{
					break;
				}
			}
		}
	}

	return sum;
}

double *SparseUndiGraph::clusteringCoefficient(double **clusteringCoeffData, int saveData)
{
	// ----------------- Init data storage --------------------------
	// determine input state
	int input_state = get_input_state(clusteringCoeffData, clusteringCoeffVector);
	// create reference to array of size n
	if (input_state == INPUT_NONE)
	{
		clusteringCoeffData = new double*[0];
		*clusteringCoeffData = new double[n];
	}

	// ------------------ computation part --------------------------
	// if no internal vector exists compute
	if (input_state == INPUT_NONE || input_state == INPUT_DATA)
	{
		double* results = *clusteringCoeffData;

		if (degreesVector == NULL)
			degrees(NULL, RES_SAVED);

		for (int i = 0; i < n; i++)
		{
			int k = degreesVector[i];
			if (k >= 2)
			{
				double a = numerOfTriangles(i);
				double ci = a/ (double) (k*k-k);
				results[i] = ci;
			}
			else
			{
				results[i] = 0;
			}

			#if (INDICATION)
				cout << "\r Clustering coeficient:" << i << "/" << data.size();
				cout.flush();
			#endif
		}
	}

	// ----------------- Storing data  --------------------------
	return close_double_method(input_state, saveData, clusteringCoeffData, clusteringCoeffVector, clusteringCoeffVectorGet, size());
}

double SparseUndiGraph::clusteringCoefficientGlobal()
{
	if (clusteringCoeffVector == NULL)
		clusteringCoefficient(NULL, RES_SAVED);

	double CC = 0;
	for (int i = 0; i < n; i++)
	{
		CC += clusteringCoeffVector[i];
	}
	CC /= n;

	return CC;
}

vector<vector<int> > *SparseUndiGraph::components(vector<vector<int> > **componentsListData, int saveData)
{
	return NULL;
}

double SparseUndiGraph::characterticPathLength()
{
	if (D == NULL)
		distanceMatrix(NULL, RES_SAVED);

	return D->meanOfNonzero(D->size() + 1);
}

int SparseUndiGraph::componentsNum()
{
	return 0;
}

double SparseUndiGraph::efficiency()
{
	// compute distance matrix if necessary
	if (D == NULL)
		distanceMatrix(NULL, RES_SAVED);

	return D->meanOfNonzeroReciprocal(D->size() + 1);
}

double SparseUndiGraph::transitivity()
{
	if (degreesVector == NULL)
		degrees(NULL, RES_SAVED);

	int denom = 0;
	double nom = 0;
	for (int i = 0; i < n; i++)
	{
		int k = degreesVector[i];

		denom += (k*k - k);

		nom += numerOfTriangles(i);
	}

	return (nom / (double) denom);
}

double SparseUndiGraph::assortativeCoefficient()
{
	return 0;
}

double *SparseUndiGraph::closeness(double **closenessData, int saveData)
{
	return NULL;
}

double *SparseUndiGraph::averageNearestNeighbor(double **annData, int saveData)
{
	return NULL;
}

int *SparseUndiGraph::componentsMat()
{
	return NULL;
}

SymMatrix<int> *SparseUndiGraph::distanceMatrix(SymMatrix<int> *dmatrix, int saveData)
{
	// ----------------- Init data storage --------------------------
	// determine input state
	int input_state = get_input_state(dmatrix, D);
	// create reference to array of size n
	if (input_state == INPUT_NONE)
	{
		dmatrix = new SymMatrix<int>(this->n);
	}

	// ------------------ computation part --------------------------
	// if no internal vector exists compute
	if (input_state == INPUT_NONE || input_state == INPUT_DATA)
	{
		// there is never path longer than this
		int infinite = size() + 1;

		//cout << "Infty is = " << infinite << endl;

		// preparation work
		// TODO Not very efficient
		SymMatrix<int> *DD = dmatrix;
		int row_start = 0;
		int row_end = n;
		int row_size = n;
		for (int i = 0; i < n; i++)
		{
			DD->setRowValue(i,infinite);
			DD->operator ()(i,i) = 0;

			int k = degreesVector[i];

			if (k != 0)
			{
				int* actneights = neighs[i];
				for (int j = 0; j < k; j++)
				{
					(*DD)(i,actneights[j]) = 1;
				}
			}
			row_size--;
			row_start += row_size;
			row_end += n;

		}

		//cout << "After initial step:" << endl;
		//cout << DD->to_s() << endl;

		int ij,ji,ik,kj;
		int irow,jrow;

		int krow = 0;
		for (int k = 0; k < n; k++)
		{
			irow = 0;
			for (int i = 0; i < n; i++)
			{
				ik = (i <= k) ? irow + k:krow + i;

				jrow = 0;
				for (int j = 0; j <= i; j++)
				{
					//ij = (i <= j) ? irow + j:jrow + i;
					ij = jrow + i;
					ji = jrow + i;
					kj = (k <= j) ? krow + j:jrow + k;

					if ((*DD)[ij] > ((*DD)[ik] + (*DD)[kj]))
					{
						//int a = (*DD)[ij];
						//int b = (*DD)[ik];
						//int c = (*DD)[kj];
						(*DD)[ij] = (*DD)[ik] + (*DD)[kj];
						(*DD)[ji] = (*DD)[ij];
						//int e =(*DD)[ij];

						//cout << "\t  " << a << " <- " << b << " + " << c << " => " << e << endl;
						//cout << "\t (" << ij << " <- " << ik << " + " << kj << ")" << endl;
					}
					jrow += (n - j - 1);
				}
				irow += (n - i - 1);
			}
			krow += (n - k - 1);
		}
	}

	//cout << "After calculation:" << endl;
	//cout << dmatrix->to_s() << endl;

	return close_symmat_int_method(input_state, saveData, &dmatrix, D, dGet, size());
}
