/*
 * Matrix.h
 *
 *  Created on: 21.3.2011
 *      Author: hartman
 */

#ifndef SYM_MATRIX_H_
#define SYM_MATRIX_H_
#include <string>
#include <sstream>
#include <iostream>
#include <math.h>
#include "../aux/StringOperations.h"
#include <cstring>
#include <vector>

template <class T>
/*!
 * 	Class representing symmetric matrix of defined type.
 */
class SymMatrix {
public:
	/*!
	 * 	Basic constructor that simply creates instance of matrix class and fill no data.
	 */
	SymMatrix();

	/*!
	 * 	Constructor that creates symmetric matrix from another one. Depending on parameter it is
	 * 	created by referencing original data or as an independent copy.
	 *
	 * 	@param lsymmat Original symmetric matrix from which a new one is created.
	 * 	@param copy Create a copy (true) or a reference (false)
	 */
	SymMatrix(SymMatrix<T> *lsymmat, bool copy = true);

	/*!
	 * 	Constructor of symmetric matrix from pure data that can be either triangular or full. During
	 * 	construction of matrix values can be put to absolute values.
	 *
	 * 	@param lmat Field representing matrix which is either symmetric and thus triangular or full,
	 *  depending on value os parameter triu.
	 *  @param lsize Size of constructed matrix
	 *  @param triu Indicator of triangular shape of input matrix.
	 *  @param absolute Indicator that force matrix creation to include transformation into absolute values
	 */
	SymMatrix(T* lmat, int lsize, bool triu, bool absolute);

	/*!
	 * 	Constructor that creates matrix of given size filled with element initElem.
	 *
	 * 	@param lsize Size of matrix
	 * 	@param initElem Value that is copied into all cells of a new matrix.
	 */
	SymMatrix(int lsize, T* initElem = NULL, T* diagElem = NULL);

	/*!
	 * 	Method to check whether internal data are specified.
	 */
	bool specified();

	/*!
	 * 	...
	 *
	 * 	@param lmat Field representing matrix which is either symmetric and thus triangular or full,
	 *  depending on value os parameter triu.
	 *  @param lsize Size of constructed matrix
	 *  @param triu Indicator of triangular shape of input matrix.
	 *  @param absolute Indicator that force matrix creation to include transformation into absolute values
	 */
	void setData(T* lmat, int lsize, bool triu, bool absolute);

	/*!
	 * 	Method to set value for given row
	 *
	 * 	@param row Index of given row
	 *  @param value Value to be set for all in row
	 */
	void setRowValue(int row, T value);

	/*!
	 * 	Method to set value for given row and column
	 *
	 * 	@param row Index of given row
	 * 	@param col Index of given col
	 *  @param value Value to be set for all in row
	 */
	void setValue(int row, int col, T value);

	/*!
	 * 	...
	 *
	 * 	@param lsymmat Original symmetric matrix from which a new one is created.
	 * 	@param copy Create a copy (true) or a reference (false)
	 */
	void setData(SymMatrix<T> *lsymmat, bool copy = true);

	/*!
	 * 	Destructor of an object.
	 */
	virtual ~SymMatrix();

	/*!
	 * 	Method to access elements in one dimension as one array, i.e. the elements are ordered in one
	 * 	array line by line (lines have decreasing sizes).
	 *
	 * 	@param i Index to access the element.
	 * 	@return Element on i-th position.
	 */
	T& operator[] (int i) const;

	/*!
	 * 	Method to access elements in matrix form, i.e. row and column index.
	 *
	 * 	@param i Row index
	 * 	@param j Column index
	 * 	@return Reference to element in i-th row and j-th column.
	 */
	T& operator() (int i, int j) const;

	/*!
	 *	Method to compute index for start of the given row
	 *
	 *	@return Index of start of the given row
	 */
	int startIndex(int row);

	/*!
	 *	Method given sum of all stored data in a matrix.
	 *
	 *	@return Sum of all data stored in matrix.
	 */
	T sumOfData();

	/*!
	 *	Method given mean of all stored data in a matrix that are below given bound.
	 *
	 *	@param upperBound Value above which the values are not considered when computing mean.
	 *	@return Mean of all data stored in matrix.
	 */
	double meanOfNonzero(T upperBound);

	/*!
	 *	Method given mean of reciprocal values of all stored data in a matrix that are below given bound.
	 *
	 *	@param upperBound Value above which the values are not considered when computing mean.
	 *	@return Mean of reciprocal values of all data stored in matrix.
	 */
	double meanOfNonzeroReciprocal(T upperBound);

	/*!
	 *	Method that gives maximum from all elmenents stored in a matrix.
	 *
	 *	@return Maximum from all elmenents stored in a matrix..
	 */
	T maxOfData();

	/*!
	 *	Method that gives stored data in a form of array. Array represents only upper triangular matrix.
	 *
	 *	@return Array containing upper triangular matrix containing the data.
	 */
	T* getData();

	/*!
	 *	Method that gives stored data in a form of array. Array represents full matrix.
	 *
	 *	@return Array containing full matrix containing the data.
	 */
	T* getFullData(T* dat = NULL);

	/*!
	 * 	Size of this symmetric square matrix, i.e. just one dimension is sufficient.
	 *
	 * 	@return Size of matrix (which is symmetric and square).
	 */
	int size();

	/*!
	 * 	Length of array containing upper triangular matrix corresponding to symmetric matrix with diagonal.
	 *
	 * 	@return Length of upper triangular matrix corresponding to data with a diagonal.
	 */
	int len();

	/*!
	 * 	Method to create a binary matrix by applying threshold t.
	 *
	 *	@param t Threshold used for binarization of the matrix.
	 * 	@return Binary matrix that contains ones only on positions where original elements were bigger than given threshold.
	 */
	SymMatrix<bool>* binarize(T t);

	/*!
	 * 	Method to return string representation of the matrix mainly for debugging purposes.
	 *
	 * 	@return String representation of the matrice.
	 */
	std::string to_s();

	/*!
	 * 	Method to return string representation of the matrix mainly for debugging purposes.
	 *
	 * 	@return String representation of the matrice.
	 */
	std::string to_s(int xmax, int ymax);

	/*!
	 * 	Clone the object.
	 *
	 * 	@return The clone of the object.
	 */
	SymMatrix<T>* clone();

protected:
	/*! Triangular data in form of field. */
	T* mat;
	/*! Size of matrix. */
	int dim;
	/*! Size of data as an array containing all elemnets.*/
	int num;

	/*! Indicator that data has been accessed and resposibility for its deletion goes to outsider */
	bool dataGet;

	void initialSetting();
};


template <typename T> SymMatrix<T>::SymMatrix(SymMatrix<T> *lsymmat, bool copy) {
	if (copy)
		memcpy(mat,lsymmat->mat,dim*sizeof(T));
	else
		mat = lsymmat->mat;

	dim = lsymmat->dim;
	num = lsymmat->num;
	initialSetting();
}

template <typename T> SymMatrix<T>::SymMatrix() {
	mat = NULL;
	dim = 0;
	num = 0;
	initialSetting();
}

template <typename T> SymMatrix<T>::SymMatrix(int lsize, T* initElem, T* diagElem)
{
	dim = lsize;
	num = lsize * (lsize + 1) / 2;
	mat = new T[num];
	if (diagElem == NULL)
	{
		if (initElem != NULL)
		{
			T c = *initElem;
			for (int i = 0; i < num; i++)
				mat[i] = c;
		}
	}
	else
	{
		if (initElem == NULL)
		{
			T d = *diagElem;
			int shift = dim;
			for (int i = 0; i < num; i += shift, shift--)
				mat[i] = d;
		}
		else
		{
			T c = *initElem;
			T d = *diagElem;
			int idx = 0;
			int shift = dim;
			for (int i = 0; i < num; i++)
			{
				if (i == idx)
				{
					mat[i] = d;
					idx += shift;
					shift--;
				}
				else
				{
					mat[i] = c;
				}
			}
		}

	}
	initialSetting();
}

template <typename T> SymMatrix<T>::SymMatrix(T* lmat, int lsize, bool triu, bool absolute) {

	dim = lsize;
	num = lsize * (lsize + 1) / 2;

	if (triu)
	{
		mat = lmat;
		if (absolute)
			for (int i = 0; i < num; i++)
				mat[i] = fabs(mat[i]);
	}
	else
	{
		mat = new T[num];
		int indx = 0;
		// TODO possible between absolute inclusion
		for (int i = 0; i < lsize; i++)
			for (int j = i; j < lsize; j++)
				mat[indx++] = (absolute) ? fabs(lmat[i*lsize + j]):lmat[i*lsize + j];
	}
	initialSetting();
}

template <typename T> void SymMatrix<T>::setData(SymMatrix<T> *lsymmat, bool copy) {
	if (copy)
		memcpy(mat,lsymmat->mat,dim*sizeof(T));
	else
		mat = lsymmat->mat;

	dim = lsymmat->dim;
	num = lsymmat->num;
	initialSetting();
}

template <typename T> void SymMatrix<T>::setData(T* lmat, int lsize, bool triu, bool absolute) {

	dim = lsize;
	num = lsize*(lsize + 1) / 2;
	if (triu)
	{
		mat = lmat;
		if (absolute)
			for (int i = 0; i < num; i++)
				mat[i] = fabs(mat[i]);
	}
	else
	{
		mat = new T[num];
		int indx = 0;
		// TODO possible between absolute inclusion
		for (int i = 0; i < lsize; i++)
			for (int j = i; j < lsize; j++)
				mat[indx++] = (absolute) ? fabs(lmat[i*lsize + j]):lmat[i*lsize + j];
	}

	initialSetting();
}

template <typename T> void SymMatrix<T>::setRowValue(int row, T value)
{
	int si = startIndex(row);
	int rowsize = dim - row;

	// TODO change
	for (int i = si; i < (si + rowsize); i++)
		mat[i] = value;
}

template <typename T> void SymMatrix<T>::setValue(int row, int col, T value)
{

}


template <typename T> void SymMatrix<T>::initialSetting()
{
	dataGet = false;
}

template <typename T> bool SymMatrix<T>::specified()
{
	return (mat != NULL);
}


template <typename T> SymMatrix<T>::~SymMatrix() {
	if (mat != NULL && !dataGet)
		delete [] mat;
}

template <typename T> T& SymMatrix<T>::operator[] (int i) const
{
	return mat[i];
}


template <typename T> T& SymMatrix<T>::operator() (int i, int j) const
{
	return (i <= j) ? (mat[i*dim - (i*(i+1)/2) + j]):(mat[j*dim - (j*(j+1)/2) + i]);
}

template <typename T> int SymMatrix<T>::startIndex(int row)
{
	// index of the beginning of the row
	return (row*dim - row*(row-1)/2);
}

template <typename T> T SymMatrix<T>::sumOfData()
{
	T sum = 0;
	for (int i = 0; i < num; i++)
		sum = sum + mat[i];
	return sum;
}

template <typename T> double SymMatrix<T>::meanOfNonzero(T upperBound)
{
	T sum = 0;
	int numNonzero = 0;
	for (int i = 0; i < num; i++)
	{
		if (mat[i] != 0 && mat[i] < upperBound)
		{
			sum = sum + mat[i];
			numNonzero++;
		}
	}
	return ((double) sum / (double) numNonzero);
}

template <typename T> double SymMatrix<T>::meanOfNonzeroReciprocal(T upperBound)
{
	double sum = 0;
	int numNonzero = 0;
	for (int i = 0; i < num; i++)
	{
		if (mat[i] != 0 && mat[i] < upperBound)
		{
			sum = sum + (1 / (double) mat[i]);
			numNonzero++;
		}
	}
	return (sum / (double) numNonzero);
}


template <typename T> T SymMatrix<T>::maxOfData()
{
	T dmax = mat[0];
	for (int i = 1; i < num; i++)
		if (dmax < mat[i])
			dmax = mat[i];

	return dmax;
}


template <typename T> T* SymMatrix<T>::getData()
{
	dataGet = true;
	return mat;
}

template <typename T> T* SymMatrix<T>::getFullData(T* dat)
{
	if (dat == NULL)
		dat = new T[dim*dim];

	T* where = dat;
	T* from = mat;
	int rowsize = dim;
	int remains = 0;
	int k = 0;
	for (int i = 0; i < dim; i++)
	{
		for (int j = 0; j < remains; j++)
			dat[k++] = (*this)(j,i);
		where += remains;

		memcpy(where, from, sizeof(T)*rowsize);

		where += rowsize;
		from += rowsize;

		k += rowsize;
		remains++;
		rowsize--;
	}

	return dat;
}

template <typename T> int SymMatrix<T>::size()
{
	return dim;
}

template <typename T> int SymMatrix<T>::len()
{
	return num;
}

template <typename T> SymMatrix<bool>* SymMatrix<T>::binarize(T t)
{
	bool* binMat = new bool[num];
	for (int i = 0; i < num; i++)
		binMat[i] = (mat[i] > t) ? 1:0;

	return new SymMatrix<bool>(binMat, dim, true, false);
}

template <typename T> std::string SymMatrix<T>::to_s()
{
	std::stringstream matstr;
	for (int i = 0; i < dim; i++)
	{
		for (int j = 0; j < dim; j++)
		{
			matstr << StringOperations::formatOutputNumber((*this)(i,j));
			matstr << "\t";
		}
		matstr << "\n";
	}
	return matstr.str();
}

template <typename T> std::string SymMatrix<T>::to_s(int xmax, int ymax)
{
	std::stringstream matstr;
	for (int i = 0; i < xmax; i++)
	{
		for (int j = 0; j < ymax; j++)
		{
			matstr << StringOperations::formatOutputNumber((*this)(i,j));
			matstr << "\t";
		}
		matstr << "\n";
	}
	return matstr.str();
}

template <typename T> SymMatrix<T>* SymMatrix<T>::clone()
{
	return new SymMatrix<T>(this, true);
}

#endif /* SYM_MATRIX_H_ */
